package cc.smarnet.escdemo;

import androidx.annotation.MainThread;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

import com.smart.command.EscCommand;
import com.smart.command.LabelCommand;
import com.smart.io.WriterResult;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Vector;

import static android.hardware.usb.UsbManager.ACTION_USB_DEVICE_ATTACHED;
import static android.hardware.usb.UsbManager.ACTION_USB_DEVICE_DETACHED;
import static cc.smarnet.escdemo.Constant.MESSAGE_UPDATE_PARAMETER;
import static cc.smarnet.escdemo.DeviceConnFactoryManager.ACTION_PRINT_FINISH;
import static cc.smarnet.escdemo.DeviceConnFactoryManager.ACTION_QUERY_PRINTER_STATE;
import static cc.smarnet.escdemo.DeviceConnFactoryManager.CONN_STATE_FAILED;

public class MainActivity extends AppCompatActivity {

    private static final String TAG = "MainActivity";
    ArrayList<String> per = new ArrayList<>();
    private UsbManager usbManager;
    private int counts;
    private static final int REQUEST_CODE = 0x004;

    /**
     * Connection status disconnected
     */
    private static final int CONN_STATE_DISCONN = 0x007;

    /**
     * Use printer command error
     */
    private static final int PRINTER_COMMAND_ERROR = 0x008;

    private static final int CONN_MOST_DEVICES = 0x11;
    private static final int CONN_PRINTER = 0x12;
    private PendingIntent mPermissionIntent;
    private String[] permissions = {
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION,
            Manifest.permission.BLUETOOTH
    };
    private String usbName;
    private TextView tvConnState;
    private ThreadPool threadPool;

    private int id = 0;
    private int printcount = 0;
    private boolean continuityprint = false;
    private DeviceConnFactoryManager deviceConnFactoryManager;

    // private KeepConn keepConn;
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Log.e(TAG, "onCreate()");
        setContentView(R.layout.activity_main);
        usbManager = (UsbManager) getSystemService(Context.USB_SERVICE);
        checkPermission();
        requestPermission();
        tvConnState = findViewById(R.id.tv_connState);
    }

    @Override
    protected void onStart() {
        super.onStart();
        IntentFilter filter = new IntentFilter();
        filter.addAction(ACTION_QUERY_PRINTER_STATE);
        filter.addAction(DeviceConnFactoryManager.ACTION_CONN_STATE);
        registerReceiver(receiver, filter);
    }

    private void checkPermission() {
        for (String permission : permissions) {
            if (PackageManager.PERMISSION_GRANTED != ContextCompat.checkSelfPermission(this, permission)) {
                per.add(permission);
            }
        }
    }

    private void requestPermission() {
        if (per.size() > 0) {
            String[] p = new String[per.size()];
            ActivityCompat.requestPermissions(this, per.toArray(p), REQUEST_CODE);
        }
    }

    /**
     * Bluetooth connect
     */
    public void btnBluetoothConn(View view) {
        startActivityForResult(new Intent(this, BluetoothDeviceList.class), Constant.BLUETOOTH_REQUEST_CODE);
    }

    /**
     * Small ticket printing
     *
     * @param view
     */
    public void btnReceiptPrint(View view) {
        if (!isConnected()) {
            Utils.toast(this, getString(R.string.str_cann_printer));
            return;
        }
        threadPool = ThreadPool.getInstantiation();
        threadPool.addTask(this::sendReceiptWithResponse);
    }

    /**
     * Disconnect
     *
     * @param view
     */
    public void btnDisConn(View view) {
        if (!isConnected()) {
            Utils.toast(this, getString(R.string.str_cann_printer));
            return;
        }
        mHandler.obtainMessage(CONN_STATE_DISCONN).sendToTarget();
    }

    public void btnPrintSelfTestPage(View view) {
        if (!isConnected()) {
            Utils.toast(this, getString(R.string.str_cann_printer));
            return;
        }
        threadPool = ThreadPool.getInstantiation();
        threadPool.addTask(() -> {
            deviceConnFactoryManager.write("SELFTEST\r\n".getBytes());
        });
    }

    public void btnQueryPower(View view) {
        if (!isConnected()) {
            Utils.toast(this, getString(R.string.str_cann_printer));
            return;
        }
        threadPool = ThreadPool.getInstantiation();
        threadPool.addTask(() -> {
            try {
                deviceConnFactoryManager.write(new byte[]{0x10, 0x06, 0x03, 0x05, 0x0c, 0x0c});
                byte[] bytes = new byte[3];
                deviceConnFactoryManager.readDataImmediately(bytes, 3, 2000);
                String power = "";
                switch (bytes[1]) {
                    case 0x0a:
                        power = getString(R.string.str_low_power);
                        break;
                    case 0x19:
                        power = getString(R.string.str_power_25);
                        break;
                    case 0x32:
                        power = getString(R.string.str_powe_50);
                        break;
                    case 0x4b:
                        power = getString(R.string.str_power_75);
                        break;
                    case 0x64:
                        power = getString(R.string.str_power_100);
                        break;
                    case (byte) 0xCB:
                        power = getString(R.string.str_charge);
                        break;
                }
                String finalPower = power;
                runOnUiThread(() -> {
                    Toast.makeText(this, finalPower, Toast.LENGTH_SHORT).show();
                });
            } catch (IOException ioException) {
                ioException.printStackTrace();
            }
        });
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode == RESULT_OK && requestCode == Constant.BLUETOOTH_REQUEST_CODE) {
            closeport();
            //Get bluetoot mac address
            String macAddress = data.getStringExtra(BluetoothDeviceList.EXTRA_DEVICE_ADDRESS);
            //Init DeviceConnFactoryManager
            deviceConnFactoryManager = new DeviceConnFactoryManager.Build()
                    .setId(id)
                    //Set bluetooth mac address
                    .setMacAddress(macAddress)
                    .build(this);
            //Open port
            threadPool = ThreadPool.getInstantiation();
            threadPool.addTask(deviceConnFactoryManager::openPort);
        }
    }

    /**
     * Reconnection recycles the last connected object to avoid memory leakage
     */
    private void closeport() {
        if (isConnected()) {
//            DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].reader.cancel();
            deviceConnFactoryManager.closePort();
        }
    }


    /**
     * Print small ticket
     */
    void sendReceiptWithResponse() {
        EscCommand esc = new EscCommand();
        esc.addInitializePrinter();
        esc.addPrintAndFeedLines((byte) 3);
        // Set print center
        esc.addSelectJustification(EscCommand.JUSTIFICATION.CENTER);
        // Set to double height and double width
        esc.addSelectPrintModes(EscCommand.FONT.FONTA, EscCommand.ENABLE.OFF, EscCommand.ENABLE.ON, EscCommand.ENABLE.ON, EscCommand.ENABLE.OFF);
        // Print text
        esc.addText("Sample\n");
        esc.addPrintAndLineFeed();

        /* Print text */
        // Cancel double height and double width
        esc.addSelectPrintModes(EscCommand.FONT.FONTA, EscCommand.ENABLE.OFF, EscCommand.ENABLE.OFF, EscCommand.ENABLE.OFF, EscCommand.ENABLE.OFF);
        // Set print left alignment
        esc.addSelectJustification(EscCommand.JUSTIFICATION.LEFT);
        // Print text
        esc.addText("Print text\n");
        // Print text
        esc.addText("Welcome to use SMARNET printer!\n");

        /* Printing traditional Chinese requires the printer to support traditional font library */
        String message = "票據打印機\n";
        esc.addText(message, "GB2312");
        esc.addPrintAndLineFeed();

        /* Please refer to programming manual for details of absolute position */
        esc.addText("Print");
        esc.addSetHorAndVerMotionUnits((byte) 7, (byte) 0);
        esc.addSetAbsolutePrintPosition((short) 6);
        esc.addText("Print");
        esc.addSetAbsolutePrintPosition((short) 10);
        esc.addText("Print");
        esc.addPrintAndLineFeed();

        /* Print picture */
        // Print text
        esc.addText("Print bitmap!\n");
        Bitmap b = BitmapFactory.decodeResource(getResources(),
                R.drawable.logo);
        // Print logo
        esc.addRasterBitImageWithMethod(b, 320, 0, 1);
        b.recycle();

        /* Printing one dimensional bar code */
        // Print text
        esc.addText("Print code128\n");
        esc.addSelectPrintingPositionForHRICharacters(EscCommand.HRI_POSITION.BELOW);
        // Set the position of bar code recognizable characters below the bar code
        // Set the bar code height to 60 points
        esc.addSetBarcodeHeight((byte) 60);
        // Set the bar code cell width to 1
        esc.addSetBarcodeWidth((byte) 1);
        // Print code128
        esc.addCODE128(esc.genCodeB("SMARNET"));
        esc.addPrintAndLineFeed();

        /*
         * QRcode command printing this command can only be used in the model that supports QRcode command printing. On the model that does not support the printing of two-dimensional code command, two-dimensional bar code image needs to be sent
         */
        // Print text
        esc.addText("Print QRcode\n");
        // Set error correction level
        esc.addSelectErrorCorrectionLevelForQRCode((byte) 0x31);
        // Set QRcode module size
        esc.addSelectSizeOfModuleForQRCode((byte) 3);
        // Set QRcode content
        esc.addStoreQRCodeData("www.smarnet.cc");
        esc.addPrintQRCode();// Print QRCode
        esc.addPrintAndLineFeed();

        // Set print left alignment
        esc.addSelectJustification(EscCommand.JUSTIFICATION.CENTER);
        //Print text
        esc.addText("Completed!\r\n");

        // Open the cash box
        esc.addGeneratePlus(LabelCommand.FOOT.F5, (byte) 255, (byte) 255);
        esc.addPrintAndFeedLines((byte) 8);

        Vector<Byte> datas = esc.getCommand();
        // send data
        deviceConnFactoryManager.sendDataImmediately(datas);
    }

    /**
     * 停止连续打印
     */
    public void btnStopContinuityPrint(View v) {
        if (deviceConnFactoryManager == null ||
                !deviceConnFactoryManager.getConnState()) {
            Utils.toast(this, getString(R.string.str_cann_printer));
            return;
        }
        if (counts != 0) {
            counts = 0;
            Utils.toast(this, getString(R.string.str_stop_continuityprint_success));
        }
    }


    private BroadcastReceiver receiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            switch (action) {
                case DeviceConnFactoryManager.ACTION_CONN_STATE:
                    int state = intent.getIntExtra(DeviceConnFactoryManager.STATE, -1);
                    int deviceId = intent.getIntExtra(DeviceConnFactoryManager.DEVICE_ID, -1);
                    switch (state) {
                        case DeviceConnFactoryManager.CONN_STATE_DISCONNECT:
                            if (id == deviceId) {
                                tvConnState.setText(getString(R.string.str_conn_state_disconnect));
                            }
                            break;
                        case DeviceConnFactoryManager.CONN_STATE_CONNECTING:
                            tvConnState.setText(getString(R.string.str_conn_state_connecting));
                            break;
                        case DeviceConnFactoryManager.CONN_STATE_CONNECTED:
                            tvConnState.setText(getString(R.string.str_conn_state_connected) + "\n" + getConnDeviceInfo());
                            break;
                        case CONN_STATE_FAILED:
                            Utils.toast(MainActivity.this, getString(R.string.str_conn_fail));
                            tvConnState.setText(getString(R.string.str_conn_state_disconnect));
                            break;
                        default:
                            break;
                    }
                    break;
                case ACTION_PRINT_FINISH:
                    Toast.makeText(context, "Print finish！", Toast.LENGTH_SHORT).show();
                    break;
                default:
                    break;
            }
        }
    };

    private final Handler mHandler = new Handler() {
        @SuppressLint("HandlerLeak")
        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case CONN_STATE_DISCONN:
                    if (deviceConnFactoryManager != null || deviceConnFactoryManager.getConnState()) {
                        deviceConnFactoryManager.closePort();
                        Utils.toast(MainActivity.this, getString(R.string.str_disconnect_success));
                    }
                    break;
                case PRINTER_COMMAND_ERROR:
                    Utils.toast(MainActivity.this, getString(R.string.str_choice_printer_command));
                    break;
                case CONN_PRINTER:
                    Utils.toast(MainActivity.this, getString(R.string.str_cann_printer));
                    break;
            }
        }
    };

    @Override
    protected void onStop() {
        super.onStop();
        unregisterReceiver(receiver);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        Log.e(TAG, "onDestroy()");
        if (isConnected()) {
            closeport();
        }
        if (threadPool != null) {
            threadPool.stopThreadPool();
        }
    }

    private boolean isConnected() {
        return deviceConnFactoryManager != null && deviceConnFactoryManager.getConnState();
    }

    private String getConnDeviceInfo() {
        String str = "";
        if (deviceConnFactoryManager != null && deviceConnFactoryManager.getConnState()) {
            str += "BLUETOOTH\n";
            str += "MacAddress: " + deviceConnFactoryManager.getMacAddress();
        }
        return str;
    }

    /**
     * Print picture
     *
     * @param view
     */
    public void btnReceiptPrintPicture(View view) {
        //check io
        if (!isConnected()) {
            Utils.toast(this, getString(R.string.str_cann_printer));
            return;
        }
        //Execute print function
        threadPool = ThreadPool.getInstantiation();
        threadPool.addTask(this::sendReceiptPrintPicture);
    }

    /**
     * Print picture
     */
    private void sendReceiptPrintPicture() {
        //Init command
        EscCommand esc = new EscCommand();
        //Loading Picture
        Bitmap b0 = BitmapFactory.decodeResource(getResources(), R.drawable.logo);
        //Ignore bitmap alpha channel
        Bitmap b0N = Bitmap.createBitmap(b0.getWidth(), b0.getHeight(), b0.getConfig());
        Canvas canvas = new Canvas(b0N);
        canvas.drawColor(Color.WHITE);
        canvas.drawBitmap(b0, 0, 0, null);
        b0.recycle();

        //Add bitmap.This function dithers the original image and transforms the dithered image data into printer image data.
        esc.addRasterBitImage(b0N, 320, 0);
        //Add esc bitmap.The function binarizes the image and transforms it into printer image data.
        esc.addRasterBitImageWithMethod(b0N, 320, 0, 1);
        b0N.recycle();
        Bitmap b1 = BitmapFactory.decodeResource(getResources(), R.drawable.picture);
        //Add bitmap.This function dithers the original image and transforms the dithered image data into printer image data.
        esc.addRasterBitImageWithMethod(640, -1, 3.7f, 0, b1);
        b1.recycle();
        Bitmap b2 = BitmapFactory.decodeResource(getResources(), R.drawable.picture_a);
        //Add bitmap.This function dithers the original image and transforms the dithered image data into printer image data.
        esc.addRasterBitImageWithMethod(640, -1, 3.7f, 0, b2);
        b2.recycle();

        //ESC Compression printing
        Bitmap b3 = BitmapFactory.decodeResource(getResources(), R.drawable.picture_a);
        //printerWidth为打印机热敏片可打印宽度。80mm打印机可打印范围为576点（或像素）。58mm打印机可打印范围为384点（或像素）
        esc.addOriginRasterBitImage(b3, 640, 576);
        esc.addPrintAndFeedLines((byte) 3);

        deviceConnFactoryManager.sendDataImmediately(esc.getCommand());
    }

    public void btnQueryPrinterStatus(View view) throws IOException {
        threadPool = ThreadPool.getInstantiation();
        threadPool.addTask(() -> {
            EscCommand esc = new EscCommand();
            esc.addQueryPrinterStatus();
            deviceConnFactoryManager.sendDataImmediately(esc.getCommand());
            byte[] buffer=  new byte[1];
            try {
                if (deviceConnFactoryManager.readDataImmediately(buffer, 1, 2000)) {
                    StringBuilder stringBuilder = new StringBuilder();
                    if (buffer[0] != 18) {
                        if (((buffer[0] & 0x4) >> 2) == 1) {
                            stringBuilder.append(getString(R.string.str_open_cover));
                            stringBuilder.append(" ");
                        }
                        if (((buffer[0] & 32) >> 5) == 1) {
                            stringBuilder.append(getString(R.string.str_out_of_paper));
                        }
                    } else {
                        stringBuilder.append(getString(R.string.str_noraml));
                    }
                    runOnUiThread(() -> {
                        Toast.makeText(this, stringBuilder, Toast.LENGTH_SHORT).show();
                    });
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        });
    }
}